﻿

namespace Hims.Api.Controllers
{
    using Hims.Api.Utilities;
    using Hims.Domain.Services;
    using Microsoft.AspNetCore.Mvc;
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Hims.Shared.UserModels.MealType;
    using Hims.Shared.DataFilters;
    using Hims.Shared.EntityModels;
    using Hims.Shared.Library.Enums;
    using Hims.Shared.UserModels.Common;
    using Hims.Api.Models;
    using Microsoft.AspNetCore.Authorization;

    /// <summary>
    /// The meal type controller.
    /// </summary>
    [Authorize]
    [Route("api/meal-types")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class MealTypesController : BaseController
    {
        /// <summary>
        /// The meal types service.
        /// </summary>
        private readonly IMealTypesService mealTypesService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public MealTypesController(IMealTypesService mealTypesService, IAuditLogService auditLogServices)
        {
            this.mealTypesService = mealTypesService;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The insert meal types masters async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("insert")]
        public async Task<ActionResult> InsertMealTypesAsync([FromBody] InsertModel model,[FromHeader] LocationHeader header)
        {
            model = (InsertModel)EmptyFilter.Handler(model);
            var response = await this.mealTypesService.AddAsync(model);
            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.MealTypes,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $"<b>{model.CreatedByName}</b> has added Meal Type : <b>{model.MealType}</b> on {DateTime.UtcNow.AddMinutes(330).ToString("dd/MM/yyyy, hh:mm tt")}",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }
     
            return response <=0 ? this.ServerError() : this.Success(response);
        }
        /// <summary>
        /// The raise indent async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchMealTypes([FromBody] ViewModel model )
        {
            model = (ViewModel)EmptyFilter.Handler(model);           
            var response = await this.mealTypesService.FetchAsync(model);
            return this.Success(response);
        }
        
        /// <summary>
        /// The raise indent async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("fetch-active")]
        public async Task<ActionResult> FetchActiveMealTypes([FromBody] ViewModel model)
        {
            model = (ViewModel)EmptyFilter.Handler(model);
            var response = await this.mealTypesService.FetchActiveMealTypes(model);
            return this.Success(response);
        }

        /// <summary>
        /// The update charge.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - charge updated successfully.
        /// - 409 - Charge already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] UpdateModel model, [FromHeader] LocationHeader header)
        {
            model = (UpdateModel)EmptyFilter.Handler(model);
            var response = await this.mealTypesService.UpdateAsync(model);
            switch (response)
            {
                case -2:
                    return this.Conflict("Given Meal Type has already been exists with us.");
                case -1:
                    return this.ServerError();
            }

            if (response > 0 )
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.MealTypes,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $"<b>{model.ModifiedByName}</b> has updated Meal Type - <b>{model.MealType}</b> on { DateTime.UtcNow.AddMinutes(330) }",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }

            return this.Success("Meal Type has been updated successfully.");
        }

        /// <summary>
        /// The delete async.
        /// </summary>
        /// <param name="model">
        /// The id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("modify-status")]
        public async Task<ActionResult> ModifyMealStatusAsync([FromBody] UpdateModel model, [FromHeader] LocationHeader header)
        {
            model = (UpdateModel)EmptyFilter.Handler(model);

            var response = await this.mealTypesService.ModifyMealStatusAsync(model);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.MealTypes,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $@"<b>{model.ModifiedByName}</b> has {((bool)model.Active ? "Activated" : "Deactivated")} the Meal Type <b> {model.MealType}</b>  successfully",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);


        }
    }
}
